//=============================================================================
// MPP_TpbTimeline.js
//=============================================================================
// Copyright (c) 2021 Mokusei Penguin
// Released under the MIT license
// http://opensource.org/licenses/mit-license.php
//=============================================================================

/*:
 * @target MZ
 * @plugindesc The accumulated condition of the time gauges of all enemies and allies is displayed on one line.
 * @author Mokusei Penguin
 * @url 
 * 
 * @help [version 1.3.0]
 * This plugin is for RPG Maker MZ.
 * 
 * ▼ Plugin command details
 *  - In MV, the variable N is referred to by writing v [N] in the item for
 *    inputting a numerical value.
 *  - In MZ, in the item to enter a numerical value, select the text and
 *    write v [N] to refer to the variable N.
 *    
 *  〇 MV / MZ
 *  
 *  〇 SetTimelineIcon actorId iconIndex  / setIcon
 *       actorId   : Actor ID
 *       iconIndex : Icon Index
 *   - Change the icon number of the actor.
 *   - The default value is the ID of each actor.
 * 
 * ▼ Enemy character note
 *  〇 <TimelineIcon:n>
 *   - Set the icon number of the timeline.
 *   - If not specified, it will be 0.
 * 
 *  〇 <TimelineCharacter:name,index>
 *   - Specifies the walking character graphic used when using automatic
 *     icon generation.
 *   - Please specify the image file name (name) and the number of
 *     character (index) to use.
 * 
 * ▼ Plugin parameter details
 *  〇 Timeline - Type
 *   - When changing to the vertical type, other parameters must be changed
 *     as well.
 *  
 * ▼ Material standard
 *  - Please put the material to be used in the img/system folder.
 *  
 *  〇 Actor icon image / Enemy character icon image
 *   - For the icon image, make one block by arranging eight in the horizontal
 *     direction, and use the block that is lengthened vertically as much as
 *     necessary.
 *   - Set the height of the icon with the plugin parameters.
 * 
 * ▼ About automatic image generation
 *  - If the timeline image and actor and enemy character icon images are
 *    not set, they will be automatically generated.
 *  - A walking character graphic is used for the actor's icon image.
 *  - If the walking character graphic is specified, the icon image of the
 *    enemy character will be used.
 * 
 * ================================
 * Mail : wood_penguin＠yahoo.co.jp (＠ is half-width)
 * Blog : http://woodpenguin.blog.fc2.com/
 * License : MIT license
 *
 *  @command setIcon
 *      @desc 
 *      @arg actorId
 *          @desc 
 *          @type actor
 *          @default 0
 *      @arg iconIndex
 *          @desc 
 *          @type number
 *          @default 0
 * 
 *  @param Timeline
 *      @desc 
 *      @type struct<Timeline>
 *      @default {"Type":"Horizontal","Image Name":"","X":"64","Y":"368","Start X":"6","Wait Width":"320","Action Width":"128"}
 * 
 *  @param Actor Icon
 *      @desc 
 *      @type struct<ActorIcon>
 *      @default {"Image Name":"","Height":"48","Anchor X":"50","Anchor Y":"15"}
 * 
 *  @param Enemy Icon
 *      @desc 
 *      @type struct<EnemyIcon>
 *      @default {"Image Name":"","Height":"48","Anchor X":"50","Anchor Y":"85","Default Character Name":"Monster","Default Character Index":"0","Symbol Type":"letter","Symbol X":"0","Symbol Y":"-15","Symbol Size":"17","Symbol Align":"left"}
 * 
 *  @param Battle Speed Rate
 *      @desc Time gauge increase rate(%)
 *      (100:Default)
 *      @type number
 *          @min 1
 *          @max 200
 *      @default 50
 * 
 *  @param Icon Stop At Force Action
 *      @desc The icon will not move while [Force Action] is running.
 *      @type boolean
 *      @default true
 * 
 */
/*~struct~Timeline:
 *  @param Type
 *      @desc 
 *      @type select
 *          @option Horizontal
 *          @option Vertical
 *      @default Horizontal
 * 
 *  @param Image Name
 *      @desc Automatically generated if not set
 *      @type file
 *          @require 1
 *          @dir img/system
 *      @default
 * 
 *  @param X
 *      @desc 
 *      @type number
 *          @min 0
 *          @max 9999
 *      @default 64
 * 
 *  @param Y
 *      @desc 
 *      @type number
 *          @min 0
 *          @max 9999
 *      @default 368
 * 
 *  @param Z
 *      @desc 
 *      @type select
 *          @option Behind the window
 *          @option Before the window
 *      @default Behind the window
 * 
 *  @param Start X
 *      @desc In the case of vertical type, Y coordinate from the bottom
 *      @type number
 *          @min 0
 *          @max 9999
 *      @default 6
 * 
 *  @param Wait Width
 *      @desc Height for vertical type
 *      @type number
 *          @min 1
 *          @max 99999
 *      @default 320
 * 
 *  @param Action Width
 *      @desc Height for vertical type
 *      @type number
 *          @min 0
 *          @max 99999
 *      @default 128
 * 
 */
/*~struct~ActorIcon:
 *  @param Image Name
 *      @desc Automatically generated if not set
 *      @type file
 *          @require 1
 *          @dir img/system
 *      @default
 *
 *  @param Height
 *      @desc Only when you specify an image
 *      @type number
 *          @min 1
 *          @max 99999
 *      @default 48
 *      @parent Image Name
 * 
 *  @param Anchor X
 *      @desc 0:Left end, 100:Right end
 *      @type number
 *          @min -50
 *          @max 150
 *      @default 50
 * 
 *  @param Anchor Y
 *      @desc 0:Top edge, 100:lower end
 *      @type number
 *          @min -50
 *          @max 150
 *      @default 15
 * 
 */
/*~struct~EnemyIcon:
 *  @param Image Name
 *      @desc Automatically generated if not set
 *      @type file
 *          @require 1
 *          @dir img/system
 *      @default
 *
 *  @param Height
 *      @desc Only when you specify an image
 *      @type number
 *          @min 1
 *          @max 99999
 *      @default 48
 *      @parent Image Name
 * 
 *  @param Anchor X
 *      @desc 0:Left end, 100:Right end
 *      @type number
 *          @min -50
 *          @max 150
 *      @default 50
 * 
 *  @param Anchor Y
 *      @desc 0:Top edge, 100:lower end
 *      @type number
 *          @min -50
 *          @max 150
 *      @default 85
 * 
 *  @param Default Character Name
 *      @desc 
 *      @type file
 *          @require 1
 *          @dir img/characters
 *      @default
 * 
 *  @param Default Character Index
 *      @desc 
 *      @type number
 *          @min 0
 *          @max 7
 *      @default 0
 * 
 *  @param Symbol Type
 *      @desc Characters to be displayed over the icon
 *      @type select
 *          @option none
 *          @option letter
 *          @option index
 *      @default letter
 * 
 *  @param Symbol X
 *      @desc 
 *      @type number
 *          @min -9999
 *          @max 9999
 *      @default 0
 * 
 *  @param Symbol Y
 *      @desc 
 *      @type number
 *          @min -9999
 *          @max 9999
 *      @default -15
 * 
 *  @param Symbol Size
 *      @desc 
 *      @type number
 *          @min 4
 *          @max 64
 *      @default 17
 * 
 *  @param Symbol Align
 *      @desc
 *      @type select
 *          @option left
 *          @option center
 *          @option right
 *      @default left
 * 
 */

/*:ja
 * @target MZ
 * @plugindesc 敵味方全員のタイムゲージのたまり具合を１ライン上に表示します。
 * @author 木星ペンギン
 * @url 
 * 
 * @help [version 1.3.0]
 * このプラグインはRPGツクールMZ用です。
 * 
 * ▼ プラグインコマンド詳細
 *  - MVでは数値を入力する項目で v[N] と記述することで変数N番を参照します。
 *  - MZでは数値を入力する項目で、テキストを選択して v[N] と記述することで
 *    変数N番を参照します。
 *    
 *  〇 MV / MZ
 *  
 *  〇 SetTimelineIcon actorId iconIndex  / アイコン設定
 *       actorId   : アクターのID
 *       iconIndex : アイコン番号
 *   - アクターのアイコン番号を変更します。
 *   - デフォルト値は各アクターのIDです。
 * 
 * ▼ 敵キャラのメモ欄
 *  〇 <TimelineIcon:n>
 *   - タイムラインのアイコン番号を設定します。
 *   - 指定していない場合は0となります。
 * 
 *  〇 <TimelineCharacter:name,index>
 *   - アイコンの自動生成を使用する際、使用される歩行キャラグラフィックを
 *     指定します。
 *   - 画像ファイル名(name)と何番目のキャラ(index)を使用するか
 *     指定してください。
 * 
 * ▼ プラグインパラメータ 詳細
 *  〇 タイムライン - タイプ
 *   - 縦タイプに変更する場合、他のパラメータも合わせて変更する必要があります。
 *  
 * ▼ 素材規格
 *  - 使用する素材は img/system フォルダに入れてください。
 *  
 *  〇 アクターアイコン画像 / 敵キャラアイコン画像
 *   - アイコン画像は横方向に8個並べたものを1ブロックとし、
 *     そのブロックを必要なだけ縦に長くしたものを使用してください。
 *   - アイコンの高さはプラグインパラメータで設定してください。
 * 
 * ▼ 画像の自動生成について
 *  - タイムライン画像とアクター及び敵キャラアイコン画像は未設定の場合、
 *    自動生成されます。
 *  - アクターのアイコン画像には歩行キャラグラフィックが使用されます。
 *  - 敵キャラのアイコン画像は、歩行キャラグラフィックを指定している場合、
 *    その画像が使用されます。
 * 
 * ================================
 * Mail : wood_penguin＠yahoo.co.jp (＠は半角)
 * Blog : http://woodpenguin.blog.fc2.com/
 * License : MIT license
 *
 *  @command setIcon
 *      @text アイコン設定
 *      @desc 
 *      @arg actorId
 *          @text アクター
 *          @desc 
 *          @type actor
 *          @default 0
 *      @arg iconIndex
 *          @text アイコン番号
 *          @desc 
 *          @type number
 *          @default 0
 * 
 *  @param Timeline
 *      @text タイムライン
 *      @desc 
 *      @type struct<Timeline>
 *      @default {"Type":"Horizontal","Image Name":"","X":"64","Y":"368","Start X":"6","Wait Width":"320","Action Width":"128"}
 * 
 *  @param Actor Icon
 *      @text アクターアイコン
 *      @desc 
 *      @type struct<ActorIcon>
 *      @default {"Image Name":"","Height":"48","Anchor X":"50","Anchor Y":"15"}
 * 
 *  @param Enemy Icon
 *      @text 敵キャラアイコン
 *      @desc 
 *      @type struct<EnemyIcon>
 *      @default {"Image Name":"","Height":"48","Anchor X":"50","Anchor Y":"85","Default Character Name":"Monster","Default Character Index":"0","Symbol Type":"letter","Symbol X":"0","Symbol Y":"-15","Symbol Size":"17","Symbol Align":"left"}
 * 
 *  @param Battle Speed Rate
 *      @text 戦闘速度
 *      @desc タイムゲージの増加率(%)
 *      (100:デフォルト)
 *      @type number
 *          @min 1
 *          @max 200
 *      @default 50
 * 
 *  @param Icon Stop At Force Action
 *      @text 戦闘行動の強制中アイコン停止
 *      @desc [戦闘行動の強制]実行中はアイコンが移動しなくなります。
 *      @type boolean
 *      @default true
 * 
 */
/*~struct~Timeline:ja
 *  @param Type
 *      @text タイプ
 *      @desc 
 *      @type select
 *          @option 横
 *          @value Horizontal
 *          @option 縦
 *          @value Vertical
 *      @default Horizontal
 * 
 *  @param Image Name
 *      @text 画像ファイル名
 *      @desc 未設定の場合は自動生成
 *      @type file
 *          @require 1
 *          @dir img/system
 *      @default
 * 
 *  @param X
 *      @desc 
 *      @type number
 *          @min 0
 *          @max 9999
 *      @default 64
 * 
 *  @param Y
 *      @desc 
 *      @type number
 *          @min 0
 *          @max 9999
 *      @default 368
 * 
 *  @param Z
 *      @desc 
 *      @type select
 *          @option ウィンドウの下
 *          @value Behind the window
 *          @option ウィンドウの上
 *          @value Before the window
 *      @default Behind the window
 * 
 *  @param Start X
 *      @text 開始X座標
 *      @desc 縦タイプの場合は下端からのY座標
 *      @type number
 *          @min 0
 *          @max 9999
 *      @default 6
 * 
 *  @param Wait Width
 *      @text ウェイトゲージ幅
 *      @desc 縦タイプの場合は高さ
 *      @type number
 *          @min 1
 *          @max 99999
 *      @default 320
 * 
 *  @param Action Width
 *      @text アクションゲージ幅
 *      @desc 縦タイプの場合は高さ
 *      @type number
 *          @min 0
 *          @max 99999
 *      @default 128
 * 
 */
/*~struct~ActorIcon:ja
 *  @param Image Name
 *      @text 画像ファイル名
 *      @desc 未設定の場合は自動生成
 *      @type file
 *          @require 1
 *          @dir img/system
 *      @default
 *
 *  @param Height
 *      @text 高さ
 *      @desc 画像を指定した場合のみ
 *      @type number
 *          @min 1
 *          @max 99999
 *      @default 48
 *      @parent Image Name
 * 
 *  @param Anchor X
 *      @text 原点X
 *      @desc 0:左端, 100:右端
 *      @type number
 *          @min -50
 *          @max 150
 *      @default 50
 * 
 *  @param Anchor Y
 *      @text 原点Y
 *      @desc 0:上端, 100:下端
 *      @type number
 *          @min -50
 *          @max 150
 *      @default 15
 * 
 */
/*~struct~EnemyIcon:ja
 *  @param Image Name
 *      @text 画像ファイル名
 *      @desc 未設定の場合は自動生成
 *      @type file
 *          @require 1
 *          @dir img/system
 *      @default
 *
 *  @param Height
 *      @text 高さ
 *      @desc 画像を指定した場合のみ
 *      @type number
 *          @min 1
 *          @max 99999
 *      @default 48
 *      @parent Image Name
 * 
 *  @param Anchor X
 *      @text 原点X
 *      @desc 0:左端, 100:右端
 *      @type number
 *          @min -50
 *          @max 150
 *      @default 50
 * 
 *  @param Anchor Y
 *      @text 原点Y
 *      @desc 0:上端, 100:下端
 *      @type number
 *          @min -50
 *          @max 150
 *      @default 85
 * 
 *  @param Default Character Name
 *      @text デフォルトキャラグラフィック
 *      @desc 
 *      @type file
 *          @require 1
 *          @dir img/characters
 *      @default
 * 
 *  @param Default Character Index
 *      @text デフォルトキャラ番号
 *      @desc 
 *      @type number
 *          @min 0
 *          @max 7
 *      @default 0
 * 
 *  @param Symbol Type
 *      @text シンボルタイプ
 *      @desc アイコンに重ねて表示する文字
 *      @type select
 *          @option none
 *          @option letter
 *          @option index
 *      @default letter
 * 
 *  @param Symbol X
 *      @text シンボルX座標
 *      @desc 
 *      @type number
 *          @min -9999
 *          @max 9999
 *      @default 0
 * 
 *  @param Symbol Y
 *      @text シンボルY座標
 *      @desc 
 *      @type number
 *          @min -9999
 *          @max 9999
 *      @default -15
 * 
 *  @param Symbol Size
 *      @text シンボル文字サイズ
 *      @desc 
 *      @type number
 *          @min 4
 *          @max 64
 *      @default 17
 * 
 *  @param Symbol Align
 *      @text シンボル揃え位置
 *      @desc
 *      @type select
 *          @option 左揃え
 *          @value left
 *          @option 中央揃え
 *          @value center
 *          @option 右揃え
 *          @value right
 *      @default left
 * 
 */

(() => {
    'use strict';

    const pluginName = 'MPP_TpbTimeline';
    
    // Plugin Parameters
    const parameters = PluginManager.parameters(pluginName);
    const paramReplace = (key, value) => {
        try {
            return JSON.parse(value);
        } catch (e) {
            return value;
        }
    };
    const param_Timeline = JSON.parse(parameters['Timeline'] || '{}', paramReplace);
    const param_ActorIcon = JSON.parse(parameters['Actor Icon'] || '{}', paramReplace);
    const param_EnemyIcon = JSON.parse(parameters['Enemy Icon'] || '{}', paramReplace);
    const param_BattleSpeedRate = Number(parameters['Battle Speed Rate'] || 50);
    const param_IconStopAtForceAction = parameters['Icon Stop At Force Action'] === 'true';
    
    // Dealing with other plugins
    const _importedPlugin = (...names) => {
        return names.some(name => PluginManager._scripts.includes(name));
    };
    
    //-------------------------------------------------------------------------
    // BattleManager

    BattleManager.isActionForcing = function() {
        return this._subject && this._action && this._action.isForcing();
    };

    //-------------------------------------------------------------------------
    // Game_Action

    Game_Action.prototype.isForcing = function() {
        return this._forcing;
    };

    //-------------------------------------------------------------------------
    // Game_Battler

    Game_Battler.prototype.tpbCastTime = function() {
        const maxTime = this.tpbRequiredCastTime();
        return maxTime > 0 ? this._tpbCastTime / maxTime : 0;
    };

    Game_Battler.prototype.isTpbCasting = function() {
        return this._tpbState === 'casting';
    };

    const _Game_Battler_tpbAcceleration = Game_Battler.prototype.tpbAcceleration;
    Game_Battler.prototype.tpbAcceleration = function() {
        const acc = _Game_Battler_tpbAcceleration.apply(this, arguments);
        return acc * param_BattleSpeedRate / 100;
    };

    const _Game_Battler_makeTpbActions = Game_Battler.prototype.makeTpbActions;
    Game_Battler.prototype.makeTpbActions = function() {
        _Game_Battler_makeTpbActions.apply(this, arguments);
        if (!this.canInput() && this._actions.every(action => !action.item())) {
            this.clearTpbChargeTime();
            this.setActionState('undecided');
        }
    };

    const _Game_Battler_onRestrict = Game_Battler.prototype.onRestrict;
    Game_Battler.prototype.onRestrict = function() {
        _Game_Battler_onRestrict.apply(this, arguments);
        this.setActionState('undecided');
    };

    //-------------------------------------------------------------------------
    // Game_Actor

    const _Game_Actor_setup = Game_Actor.prototype.setup;
    Game_Actor.prototype.setup = function(actorId) {
        _Game_Actor_setup.apply(this, arguments);
        this._timelineIconIndex = actorId;
    };

    Game_Actor.prototype.timelineIconIndex = function() {
        return this._timelineIconIndex;
    };

    Game_Actor.prototype.setTimelineIcon = function(index) {
        this._timelineIconIndex = index;
    };

    //-------------------------------------------------------------------------
    // Game_Enemy

    Game_Enemy.prototype.timelineIconIndex = function() {
        const index = this.enemy().meta.TimelineIcon;
        return index ? Number(index) : 0;
    };

    Game_Enemy.prototype.timelineSymbol = function() {
        switch (param_EnemyIcon['Symbol Type']) {
            case 'letter':
                return this._plural ? this._letter : '';
            case 'index':
                return $gameTroop.members().length > 1 ? this.index() + 1 : '';
        }
        return '';
    };

    Game_Enemy.prototype.timelineCharacter = function() {
        const character = this.enemy().meta.TimelineCharacter;
        if (character) {
            const [ name, index ] = character.split(',');
            return [name, Number(index)];
        }
        const characterName = param_EnemyIcon['Default Character Name'];
        if (characterName) {
            const characterIndex = param_EnemyIcon['Default Character Index'] || 0;
            return [characterName, characterIndex];
        }
        return [];
    };

    //-------------------------------------------------------------------------
    // Game_Interpreter

    const _mzCommands = {
        SetTimelineIcon: { name:'setIcon', keys:['actorId', 'iconIndex'] }
    };
    Object.assign(_mzCommands, {
        'タイムラインアイコン設定': _mzCommands.SetTimelineIcon
    });

    const _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function(command, args) {
        _Game_Interpreter_pluginCommand.apply(this, arguments);
        const mzCommand = _mzCommands[command];
        if (mzCommand) {
            const args2 = Object.assign(...mzCommand.keys.map((k,i) => ({[k]:args[i]})));
            PluginManager.callCommand(this, pluginName, mzCommand.name, args2);
        }
    };
    
    //-------------------------------------------------------------------------
    // PluginManager
    
    PluginManager._commands = PluginManager._commands || {};
    
    if (!PluginManager.registerCommand) {
        PluginManager.registerCommand = function(pluginName, commandName, func) {
            const key = pluginName + ":" + commandName;
            this._commands[key] = func;
        };
    }

    if (!PluginManager.callCommand) {
        PluginManager.callCommand = function(self, pluginName, commandName, args) {
            const key = pluginName + ":" + commandName;
            const func = this._commands[key];
            if (typeof func === "function") {
                func.bind(self)(args);
            }
        };
    }

    PluginManager.registerCommand(pluginName, 'setIcon', args => {
        const actorId = PluginManager.mppValue(args.actorId);
        const actor = $gameActors.actor(actorId);
        if (actor) {
            actor.setTimelineIcon(PluginManager.mppValue(args.iconIndex));
        }
    });

    PluginManager.mppValue = function(value) {
        const match = /^V\[(\d+)\]$/i.exec(value);
        return match ? $gameVariables.value(+match[1]) : +value;
    };
    
    //-----------------------------------------------------------------------------
    // Sprite
    
    if (!_importedPlugin('MPP_PatchMZ')) {
        
        Sprite.prototype._removeColorFilter = function() {
            if (this._colorFilter) {
                this.filters.remove(this._colorFilter);
                this._colorFilter = null;
            }
        };

        const _Sprite__updateColorFilter = Sprite.prototype._updateColorFilter;
        Sprite.prototype._updateColorFilter = function() {
            if (this._hue === 0 && this._blendColor[3] === 0 &&
                    this._colorTone.equals([0, 0, 0, 0])) {
                this._removeColorFilter();
            } else {
                _Sprite__updateColorFilter.apply(this, arguments);
            }
        };
        
    }

    //-------------------------------------------------------------------------
    // Sprite_TimelineIcon

    function Sprite_TimelineIcon() {
        this.initialize.apply(this, arguments);
    }
    
    if (_importedPlugin('MPP_TpbTimeline_Op1')) {
        window.Sprite_TimelineIcon = Sprite_TimelineIcon;
    }

    Sprite_TimelineIcon.prototype = Object.create(Sprite.prototype);
    Sprite_TimelineIcon.prototype.constructor = Sprite_TimelineIcon;

    Sprite_TimelineIcon.prototype.initialize = function(battler) {
        Sprite.prototype.initialize.call(this);
        this.initMembers();
        this.z = this.basePriority();
        this.setBattler(battler);
        this.updateAnchor();
    };

    Sprite_TimelineIcon.prototype.initMembers = function() {
        this._battler = null;
        this._iconSymbol = '';
        this._value = NaN;
        this._targetValue = NaN;
        this._duration = 0;
        this._flashingCount = 0;
        this._original = !!this.iconParam()['Image Name'];
        this._characterBitmap = null;
        this._characterName = null;
        this._characterParams = null;
    };

    Sprite_TimelineIcon.prototype.destroy = function(options) {
        if (!this._original && this.bitmap) {
            this.bitmap.destroy();
        }
        Sprite.prototype.destroy.call(this, options);
    };

    Sprite_TimelineIcon.prototype.basePriority = function() {
        return 0;
    };

    Sprite_TimelineIcon.prototype.iconParam = function() {
        return null;
    };

    Sprite_TimelineIcon.prototype.setBattler = function(battler) {
        if (this._battler !== battler) {
            this._battler = battler;
            this._value = this.currentValue();
            if (!this._original && this.bitmap) {
                this.bitmap.destroy();
            }
            this.bitmap = this.iconBitmap();
            if (battler) this.updateFrame();
            this.updateMove();
        }
    };

    Sprite_TimelineIcon.prototype.iconBitmap = function() {
        if (this._battler) {
            return this.userIconBitmap() || this.createBitmap();
        }
        return null;
    };

    Sprite_TimelineIcon.prototype.userIconBitmap = function() {
        const imageName = this.iconParam()['Image Name'] || '';
        return imageName ? ImageManager.loadSystem(imageName) : null;
    };

    Sprite_TimelineIcon.prototype.createBitmap = function() {
        if (param_Timeline.Type === 'Horizontal') {
            return this.createHorizontalBitmap();
        } else {
            return this.createVerticalBitmap();
        }
    };

    Sprite_TimelineIcon.prototype.createHorizontalBitmap = function() {
        const { 'Anchor Y': anchorY = 50 } = this.iconParam();
        const bitmap = new Bitmap(34, 48);
        const x = 17;
        const y = 24 + (anchorY < 25 ? 6 : anchorY > 75 ? -6 : 0);
        if (anchorY < 25) {
            this.drawUpTriangle(bitmap, x, y);
        } else if (anchorY > 75) {
            this.drawDownTriangle(bitmap, x, y);
        }
        this.drawSquare(bitmap, x, y);
        this.setCharacterBitmap(x, y);
        return bitmap;
    };

    Sprite_TimelineIcon.prototype.createVerticalBitmap = function() {
        const { 'Anchor X': anchorX = 50 } = this.iconParam();
        const bitmap = new Bitmap(48, 36);
        const x = 24 + (anchorX < 25 ? 6 : anchorX > 75 ? -6 : 0);
        const y = 18;
        if (anchorX < 25) {
            this.drawLeftTriangle(bitmap, x, y);
        } else if (anchorX > 75) {
            this.drawRightTriangle(bitmap, x, y);
        }
        this.drawSquare(bitmap, x, y);
        this.setCharacterBitmap(x, y);
        return bitmap;
    };

   Sprite_TimelineIcon.prototype.drawUpTriangle = function(bitmap, x, y) {
        const { 'Anchor X': anchorX = 50 } = this.iconParam();
        const sx = bitmap.width * (anchorX - 50) / 100;
        const context = bitmap.context;
        context.save();
        context.beginPath();
        context.moveTo(x + sx, y - 30);
        context.lineTo(x + 15, y);
        context.lineTo(x - 15, y);
        context.closePath();
        this._paintTriangle(context, x, y - 30, x, y);
        context.restore();
    };

    Sprite_TimelineIcon.prototype.drawDownTriangle = function(bitmap, x, y) {
        const { 'Anchor X': anchorX = 50 } = this.iconParam();
        const sx = bitmap.width * (anchorX - 50) / 100;
        const context = bitmap.context;
        context.save();
        context.beginPath();
        context.moveTo(x + sx, y + 30);
        context.lineTo(x - 15, y);
        context.lineTo(x + 15, y);
        context.closePath();
        this._paintTriangle(context, x, y + 30, x, y);
        context.restore();
    };

    Sprite_TimelineIcon.prototype.drawLeftTriangle = function(bitmap, x, y) {
        const { 'Anchor Y': anchorY = 50 } = this.iconParam();
        const sy = bitmap.height * (anchorY - 50) / 100;
        const context = bitmap.context;
        context.save();
        context.beginPath();
        context.moveTo(x - 30, y + sy);
        context.lineTo(x, y - 15);
        context.lineTo(x, y + 15);
        context.closePath();
        this._paintTriangle(context, x - 30, y, x, y);
        context.restore();
    };

    Sprite_TimelineIcon.prototype.drawRightTriangle = function(bitmap, x, y) {
        const { 'Anchor Y': anchorY = 50 } = this.iconParam();
        const sy = bitmap.height * (anchorY - 50) / 100;
        const context = bitmap.context;
        context.save();
        context.beginPath();
        context.moveTo(x + 30, y + sy);
        context.lineTo(x, y + 15);
        context.lineTo(x, y - 15);
        context.closePath();
        this._paintTriangle(context, x + 30, y, x, y);
        context.restore();
    };

    Sprite_TimelineIcon.prototype._paintTriangle = function(context, x1, y1, x2, y2) {
        const gradient = context.createLinearGradient(x1, y1, x2, y2);
        gradient.addColorStop(0, this.triangleColor1());
        gradient.addColorStop(1, this.triangleColor2());
        context.fillStyle = gradient;
        context.fill();
        context.lineJoin = 'bevel';
        context.strokeStyle = 'black';
        context.stroke();
    };

    Sprite_TimelineIcon.prototype.triangleColor1 = function() {
        return null;
    };
    
    Sprite_TimelineIcon.prototype.triangleColor2 = function() {
        return null;
    };
    
    Sprite_TimelineIcon.prototype.drawSquare = function(bitmap, x, y) {
        const context = bitmap.context;
        const radius = 15;
        context.save();
        context.beginPath();
        context.moveTo(x, y - radius);
        context.lineTo(x + radius, y);
        context.lineTo(x, y + radius);
        context.lineTo(x - radius, y);
        context.closePath();
        context.fillStyle = 'white';
        context.fill();
        context.lineWidth = 4;
        context.lineJoin = 'bevel';
        context.strokeStyle = 'black';
        context.stroke();
        const gradient = context.createLinearGradient(x, y - 16, x, y + 16);
        gradient.addColorStop(0, this.squareColor1());
        gradient.addColorStop(1, this.squareColor2());
        context.lineWidth = 2;
        context.strokeStyle = gradient;
        context.stroke();
        context.restore();
    };

    Sprite_TimelineIcon.prototype.squareColor1 = function() {
        return null;
    };
    
    Sprite_TimelineIcon.prototype.squareColor2 = function() {
        return null;
    };
    
    Sprite_TimelineIcon.prototype.drawCharacter = function(charBitmap, param) {
        const { index, x, y, isBig } = param;
        const r = 17;
        const context = this.bitmap.context;
        context.save();
        context.beginPath();
        context.moveTo(x - r, y - r);
        context.lineTo(x + r, y - r);
        context.lineTo(x + r, y);
        context.lineTo(x + 13, y);
        context.lineTo(x, y + 13);
        context.lineTo(x - 13, y);
        context.lineTo(x - r, y);
        context.closePath();
        context.clip();
        const pw = charBitmap.width / (isBig ? 3 : 12);
        const ph = charBitmap.height / (isBig ? 4 : 8);
        const sx = ((isBig ? 0 : (index % 4) * 3) + 1) * pw;
        const sy = ((isBig ? 0 : Math.floor(index / 4) * 4) + 2) * ph;
        const dx = x - r;
        const dy = y - r;
        const dw = r * 2;
        const dh = Math.floor(ph * dw / pw);
        context.drawImage(charBitmap.canvas, sx, sy, pw, ph, dx, dy, dw, dh);
        context.restore();
    };

    Sprite_TimelineIcon.prototype.setCharacterBitmap = function(x, y) {
    };

    Sprite_TimelineIcon.prototype.updateAnchor = function() {
        const {
            'Anchor X': anchorX = 50,
            'Anchor Y': anchorY = 50
        } = this.iconParam();
        this.anchor.x = anchorX / 100;
        this.anchor.y = anchorY / 100;
    };

    Sprite_TimelineIcon.prototype.update = function() {
        Sprite.prototype.update.call(this);
        this.updateVisibility();
        if (this.visible && this.bitmap.isReady()) {
            this.updateCharacter();
            this.updateFrame();
            this.updateMove();
            this.updateFlashing();
            this.updatePriority();
        }
    };

    Sprite_TimelineIcon.prototype.updateCharacter = function() {
        if (this._characterBitmap && this._characterBitmap.isReady()) {
            this.drawCharacter(this._characterBitmap, this._characterParams);
            this._characterBitmap = null;
        }
    };

    Sprite_TimelineIcon.prototype.updateVisibility = function() {
        const battler = this._battler;
        this.visible = (battler && (battler.isActor() || battler.isAlive()));
    };

    Sprite_TimelineIcon.prototype.updateFrame = function() {
        if (this._original) {
            const iconIndex = this._battler.timelineIconIndex();
            const cw = this.bitmap.width / 8;
            const ch = this.iconParam()['Height'] || 48;
            const cx = iconIndex % 8 * cw;
            const cy = Math.floor(iconIndex / 8) * ch;
            this.setFrame(cx, cy, cw, ch);
        }
    };

    Sprite_TimelineIcon.prototype.updateMove = function() {
        const value = this.currentValue();
        if (value !== this._targetValue) {
            this.updateTargetValue(value);
        }
        this.updateAnimation();
        this.updatePosition();
    };

    Sprite_TimelineIcon.prototype.updateTargetValue = function(value) {
        this._targetValue = value;
        if (isNaN(this._value)) {
            this._value = value;
        } else {
            this._duration = this._value < value ? 4 : 8;
        }
    };

    Sprite_TimelineIcon.prototype.updateAnimation = function() {
        if (this._duration > 0) {
            const d = this._duration;
            this._value = (this._value * (d - 1) + this._targetValue) / d;
            this._duration--;
        }
    };

    Sprite_TimelineIcon.prototype.updatePosition = function() {
        const {
            'Type': type = 'Horizontal',
            'Start X': startX = 6
        } = param_Timeline;
        const timeWidth = this.timeWidth();
        if (type === 'Horizontal') {
            this.x = startX + timeWidth;
            this.y = 0;
        } else {
            this.x = 0;
            this.y = -startX - timeWidth;
        }
    };

    Sprite_TimelineIcon.prototype.timeWidth = function() {
        const {
            'Wait Width': waitWidth = 320,
            'Action Width': actionWidth = 128
        } = param_Timeline;
        if (this._value > 1) {
            return waitWidth + Math.floor(actionWidth * (this._value - 1));
        } else {
            return Math.floor(waitWidth * this._value);
        }
    };

    Sprite_TimelineIcon.prototype.currentValue = function() {
        const battler = this._battler;
        if (battler) {
            if (battler.isTpbCasting()) {
                return battler.tpbCastTime() + 1;
            } else if (this.isBattlerActing(battler)) {
                return 2;
            } else {
                return battler.tpbChargeTime();
            }
        }
        return NaN;
    };

    Sprite_TimelineIcon.prototype.isBattlerActing = function(battler) {
        return (battler.isWaiting() || battler.isActing()) && !this.isEventWait();
    };

    Sprite_TimelineIcon.prototype.isEventWait = function() {
        return param_IconStopAtForceAction && BattleManager.isActionForcing();
    };

    Sprite_TimelineIcon.prototype.updateFlashing = function() {
        this._flashingCount++;
        if (this._battler.isInputting() || this._battler.isSelected()) {
            if (this._flashingCount % 30 < 15) {
                this.setBlendColor(this.flashingColor1());
            } else {
                this.setBlendColor(this.flashingColor2());
            }
        } else {
            this.setBlendColor([0, 0, 0, 0]);
        }
    };

    Sprite_TimelineIcon.prototype.flashingColor1 = function() {
        return [255, 255, 255, 64];
    };

    Sprite_TimelineIcon.prototype.flashingColor2 = function() {
        return this._battler.isSelected() ? [0, 0, 0, 0] : [0, 0, 255, 48];
    };

    Sprite_TimelineIcon.prototype.updatePriority = function() {
        this.z = this._battler.isSelected()
            ? 3
            : this._battler.isInputting() ? 2 : this.basePriority();
    };

    //-------------------------------------------------------------------------
    // Sprite_ActorTimelineIcon

    function Sprite_ActorTimelineIcon() {
        this.initialize.apply(this, arguments);
    }

    if (_importedPlugin('MPP_TpbTimeline_Op1')) {
        window.Sprite_ActorTimelineIcon = Sprite_ActorTimelineIcon;
    }

    Sprite_ActorTimelineIcon.prototype = Object.create(Sprite_TimelineIcon.prototype);
    Sprite_ActorTimelineIcon.prototype.constructor = Sprite_ActorTimelineIcon;

    Sprite_ActorTimelineIcon.prototype.initialize = function(battler) {
        Sprite_TimelineIcon.prototype.initialize.call(this, battler);
    };

    Sprite_ActorTimelineIcon.prototype.basePriority = function() {
        return 1;
    };

    Sprite_ActorTimelineIcon.prototype.iconParam = function() {
        return param_ActorIcon;
    };

    Sprite_ActorTimelineIcon.prototype.triangleColor1 = function() {
        return 'rgb(255,255,255)';
    };
    
    Sprite_ActorTimelineIcon.prototype.triangleColor2 = function() {
        return 'rgb(128,255,255)';
    };
    
    Sprite_ActorTimelineIcon.prototype.squareColor1 = function() {
        return 'rgb(128,255,255)';
    };
    
    Sprite_ActorTimelineIcon.prototype.squareColor2 = function() {
        return 'rgb(0,128,255)';
    };
    
    Sprite_ActorTimelineIcon.prototype.setCharacterBitmap = function(x, y) {
        const characterName = this._battler.characterName();
        const index = this._battler.characterIndex();
        const isBig = ImageManager.isBigCharacter(characterName);
        this._characterBitmap = ImageManager.loadCharacter(characterName);
        this._characterParams = { index, x, y, isBig };
    };

    //-------------------------------------------------------------------------
    // Sprite_EnemyTimelineIcon

    function Sprite_EnemyTimelineIcon() {
        this.initialize.apply(this, arguments);
    }

    if (_importedPlugin('MPP_TpbTimeline_Op1')) {
        window.Sprite_EnemyTimelineIcon = Sprite_EnemyTimelineIcon;
    }

    Sprite_EnemyTimelineIcon.prototype = Object.create(Sprite_TimelineIcon.prototype);
    Sprite_EnemyTimelineIcon.prototype.constructor = Sprite_EnemyTimelineIcon;

    Sprite_EnemyTimelineIcon.prototype.initialize = function(battler) {
        Sprite_TimelineIcon.prototype.initialize.call(this, battler);
        this.createSymbolSprite();
    };

    Sprite_EnemyTimelineIcon.prototype.destroy = function(options) {
        if (this._symbolSprite.bitmap) {
            this._symbolSprite.bitmap.destroy();
        }
        this._symbolSprite.destroy();
        Sprite_TimelineIcon.prototype.destroy.call(this, options);
    };

    Sprite_EnemyTimelineIcon.prototype.createSymbolSprite = function() {
        this._symbolSprite = new Sprite();
        this._symbolSprite.opacity = 224;
        this._symbolSprite.anchor.x = 0.5;
        this._symbolSprite.anchor.y = 0.5;
        this.addChild(this._symbolSprite);
    };

    Sprite_EnemyTimelineIcon.prototype.iconParam = function() {
        return param_EnemyIcon;
    };

    Sprite_EnemyTimelineIcon.prototype.triangleColor1 = function() {
        return 'rgb(255,255,255)';
    };
    
    Sprite_EnemyTimelineIcon.prototype.triangleColor2 = function() {
        return 'rgb(255,192,192)';
    };
    
    Sprite_EnemyTimelineIcon.prototype.squareColor1 = function() {
        return 'rgb(255,192,192)';
    };
    
    Sprite_EnemyTimelineIcon.prototype.squareColor2 = function() {
        return 'rgb(255,96,96)';
    };
    
    Sprite_EnemyTimelineIcon.prototype.setCharacterBitmap = function(x, y) {
        const [ characterName, index ] = this._battler.timelineCharacter();
        if (characterName) {
            const isBig = ImageManager.isBigCharacter(characterName);
            this._characterBitmap = ImageManager.loadCharacter(characterName);
            this._characterParams = { index, x, y, isBig };
        }
        this.setSymbolPosition();
    };

    Sprite_EnemyTimelineIcon.prototype.setSymbolPosition = function() {
        const {
            'Symbol X': symbolX = 0,
            'Symbol Y': symbolY = 0
        } = this.iconParam();
        this._symbolSprite.x = symbolX;
        this._symbolSprite.y = symbolY;
    };

    Sprite_EnemyTimelineIcon.prototype.update = function() {
        Sprite_TimelineIcon.prototype.update.call(this);
        if (this.visible && this.bitmap.isReady()) {
            this.updateSymbol();
        }
    };

    Sprite_EnemyTimelineIcon.prototype.updateSymbol = function() {
        const symbol = this._battler.timelineSymbol();
        if (this._iconSymbol !== symbol) {
            this._iconSymbol = symbol;
            if (symbol) {
                if (!this._symbolSprite.bitmap) {
                    this._symbolSprite.bitmap = this.createSymbolBitmap();
                }
                this.redrawSymbol(symbol);
                this._symbolSprite.visible = true;
            } else {
                this._symbolSprite.visible = false;
            }
        }
    };

    Sprite_EnemyTimelineIcon.prototype.createSymbolBitmap = function() {
        const min = Math.min(this.width, this.height);
        const size = this.iconParam()['Symbol Size'] || 17;
        const bitmap = new Bitmap(min, size + 4);
        bitmap.fontSize = size;
        bitmap.fontBold = true;
        bitmap.outlineColor = 'rgba(0, 0, 0, 0.8)';
        bitmap.outlineWidth = 4;
        return bitmap;
    };

    Sprite_EnemyTimelineIcon.prototype.redrawSymbol = function(symbol) {
        const bitmap = this._symbolSprite.bitmap;
        bitmap.clear();
        const align = this.iconParam()['Symbol Align'] || 'right';
        bitmap.drawText(symbol, 2, 0, bitmap.width - 4, bitmap.height, align);
    };

    //-----------------------------------------------------------------------------
    // Sprite_TimelineBar

    function Sprite_TimelineBar() {
        this.initialize.apply(this, arguments);
    }

    if (_importedPlugin('MPP_TpbTimeline_Op1')) {
        window.Sprite_TimelineBar = Sprite_TimelineBar;
    }

    Sprite_TimelineBar.prototype = Object.create(Sprite.prototype);
    Sprite_TimelineBar.prototype.constructor = Sprite_TimelineBar;

    Sprite_TimelineBar.prototype.initialize = function() {
        Sprite.prototype.initialize.call(this);
        this.initMembers();
        this.createWaitBarSprite();
        this.createActionBarSprite();
        this.createMainSprite();
        this.updatePosition();
    };

    Sprite_TimelineBar.prototype.destroy = function(options) {
        if (this._waitBarSprite) {
            this._waitBarSprite.bitmap.destroy();
            this._waitBarSprite.destroy();
        }
        if (this._actionBarSprite) {
            this._actionBarSprite.bitmap.destroy();
            this._actionBarSprite.destroy();
        }
        if (!this._original) {
            this._mainSprite.bitmap.destroy();
        }
        this._mainSprite.destroy();
        Sprite_TimelineIcon.prototype.destroy.call(this, options);
    };

    Sprite_TimelineBar.prototype.initMembers = function() {
        this._original = !!param_Timeline['Image Name'];
        this._animationCount = 0;
        this.z = -1;
    };

    Sprite_TimelineBar.prototype.createWaitBarSprite = function() {
        if (!this._original) {
            const { 'Start X': startX = 6 } = param_Timeline;
            this._waitBarSprite = new Sprite();
            this._waitBarSprite.bitmap = this.createWaitBarBitmap();
            this._waitBarSprite.x = 2;
            this._waitBarSprite.anchor.y = 0.5;
            this._waitBarSprite.opacity = 192;
            this.addChild(this._waitBarSprite);
            this.updateWaitBarFrame();
        }
    };

    Sprite_TimelineBar.prototype.createWaitBarBitmap = function() {
        const {
            'Start X': startX = 6,
            'Wait Width': waitWidth = 320,
            'Action Width': actionWidth = 128
        } = param_Timeline;
        const pad = (startX - 2) * (actionWidth === 0 ? 2 : 1);
        const width = Math.ceil((pad + waitWidth) / 2) * 3;
        const bitmap = new Bitmap(width, 20);
        this.drawBitmapWaitBar(bitmap);
        return bitmap;
    };

    Sprite_TimelineBar.prototype.drawBitmapWaitBar = function(bitmap) {
        const context = bitmap.context;
        const color1 = 'rgb(128,160,255)';
        const color2 = 'rgb(224,224,255)';
        const grad = context.createLinearGradient(0, 0, bitmap.width, 0);
        for (let i = 0; i <= 6; i++) {
            grad.addColorStop(i / 6, i % 2 === 0 ? color1 : color2);
        }
        context.save();
        context.fillStyle = grad;
        context.fillRect(0, 0, bitmap.width, bitmap.height);
        context.restore();
    };

    Sprite_TimelineBar.prototype.createActionBarSprite = function() {
        const {
            'Start X': startX = 6,
            'Wait Width': waitWidth = 320,
            'Action Width': actionWidth = 128
        } = param_Timeline;
        if (!this._original && actionWidth > 0) {
            this._actionBarSprite = new Sprite();
            this._actionBarSprite.bitmap = this.createActionBarBitmap();
            this._actionBarSprite.x = startX + waitWidth;
            this._actionBarSprite.anchor.y = 0.5;
            this._actionBarSprite.opacity = 192;
            this.addChild(this._actionBarSprite);
            this.updateActionBarFrame();
        }
    };

    Sprite_TimelineBar.prototype.createActionBarBitmap = function() {
        const {
            'Start X': startX = 6,
            'Action Width': actionWidth = 128
        } = param_Timeline;
        const width = Math.ceil((startX + actionWidth - 2) * 5 / 3);
        const bitmap = new Bitmap(width, 20);
        this.drawBitmapActionBar(bitmap, startX + actionWidth - 2);
        return bitmap;
    };

    Sprite_TimelineBar.prototype.drawBitmapActionBar = function(bitmap, width) {
        const context = bitmap.context;
        context.save();
        context.transform(1, 0, 1, 1, -20, 0);
        const sectionWidth = Math.ceil(width * 2 / 3);
        const color1 = 'rgb(192,32,0)';
        const color2 = 'rgb(255,192,128)';
        for (let i = 0; i < 3; i++) {
            const gx1 = sectionWidth * i;
            const grad = context.createLinearGradient(gx1, 0, gx1 + sectionWidth, 0);
            grad.addColorStop(0, color1);
            grad.addColorStop(1, color2);
            context.fillStyle = grad;
            context.fillRect(gx1, 0, sectionWidth, bitmap.height);
        }
        context.restore();
    };

    Sprite_TimelineBar.prototype.createMainSprite = function() {
        this._mainSprite = new Sprite();
        this._mainSprite.bitmap = this.timelineBitmap();
        this._mainSprite.anchor.y = 0.5;
        this.addChild(this._mainSprite);
    };

    Sprite_TimelineBar.prototype.timelineBitmap = function() {
        const imageName = param_Timeline['Image Name'];
        return this._original
            ? ImageManager.loadSystem(imageName)
            : this.createMainBitmap();
    };

    Sprite_TimelineBar.prototype.createMainBitmap = function() {
        const {
            'Start X': startX = 6,
            'Wait Width': waitWidth = 320,
            'Action Width': actionWidth = 128
        } = param_Timeline;
        const width = startX * 2 + waitWidth + actionWidth;
        const bitmap = new Bitmap(width, 48);
        this.drawBitmapBack(bitmap, 24);
        this.drawBitmapLine(bitmap, startX);
        this.drawBitmapLine(bitmap, startX + waitWidth);
        if (actionWidth > 0) {
            this.drawBitmapLine(bitmap, startX + waitWidth + actionWidth);
        }
        this.resetFontSettings(bitmap);
        bitmap.textColor = 'rgb(0,0,128)';
        this.drawBitmapText(bitmap, 'WAIT', startX + 8);
        if (actionWidth > 32) {
            bitmap.textColor = 'rgb(128,0,0)';
            const text = actionWidth >= 88 ? 'ACTION' : 'ACT';
            this.drawBitmapText(bitmap, text, startX + waitWidth + 8);
        }
        return bitmap;
    };

    Sprite_TimelineBar.prototype.drawBitmapBack = function(bitmap, bh) {
        const { context, width, height } = bitmap;
        context.save();
        context.lineWidth = 2;
        context.strokeStyle = 'rgba(32,32,64,0.5)';
        context.strokeRect(1, (height - bh) / 2 + 1, width - 2, bh - 2);
        context.restore();
    };

    Sprite_TimelineBar.prototype.drawBitmapLine = function(bitmap, x) {
        const color1 = 'rgba(255,255,255,0.25)';
        const color2 = 'rgba(255,255,255,1.0)';
        const context = bitmap.context;
        const grad = context.createLinearGradient(x, 0, x, bitmap.height);
        grad.addColorStop(0, color1);
        grad.addColorStop(0.5, color2);
        grad.addColorStop(1, color1);
        context.save();
        context.fillStyle = grad;
        context.fillRect(x - 1, 0, 2, bitmap.height);
        context.restore();
    };

    Sprite_TimelineBar.prototype.resetFontSettings = function(bitmap) {
        bitmap.fontSize = 18;
        bitmap.fontBold = true;
        bitmap.fontItalic = true;
        bitmap.outlineColor = 'rgba(255,255,255,0.9)';
        bitmap.outlineWidth = 5;
    };

    Sprite_TimelineBar.prototype.drawBitmapText = function(bitmap, text, x) {
        bitmap.drawText(text, x, 10, 128, 30);
    };

    Sprite_TimelineBar.prototype.updatePosition = function() {
        this.anchor.y = 0.5;
        this.rotation = this._original || param_Timeline.Type === 'Horizontal'
            ? 0
            : 270 * Math.PI / 180;
    };

    Sprite_TimelineBar.prototype.update = function() {
        Sprite.prototype.update.call(this);
        this.updateAnimation();
    };

    Sprite_TimelineBar.prototype.updateAnimation = function() {
        if (!this._original) {
            this._animationCount++;
            this.updateWaitBarFrame();
            this.updateActionBarFrame();
        }
    };

    Sprite_TimelineBar.prototype.updateWaitBarFrame = function() {
        const {
            'Start X': startX = 6,
            'Wait Width': waitWidth = 320,
            'Action Width': actionWidth = 128
        } = param_Timeline;
        const c = this._animationCount % 180;
        const width = (startX - 2) * (actionWidth === 0 ? 2 : 1) + waitWidth;
        const x = width / 2 * (179 - c) / 180;
        this._waitBarSprite.setFrame(x, 0, width, 20);
    };

    Sprite_TimelineBar.prototype.updateActionBarFrame = function() {
        if (this._actionBarSprite) {
            const {
                'Start X': startX = 4,
                'Action Width': actionWidth = 128
            } = param_Timeline;
            const c = this._animationCount % 60;
            const width = startX + actionWidth - 2;
            const x = Math.ceil(width * 2 / 3) * (59 - c) / 60;
            this._actionBarSprite.setFrame(x, 0, width, 20);
        }
    };

    //-----------------------------------------------------------------------------
    // Sprite_Timeline

    function Sprite_Timeline() {
        this.initialize.apply(this, arguments);
    }

    Sprite_Timeline.prototype = Object.create(Sprite.prototype);
    Sprite_Timeline.prototype.constructor = Sprite_Timeline;

    Sprite_Timeline.prototype.initialize = function() {
        Sprite.prototype.initialize.call(this);
        this.createTimelineBarSprite();
        this.createEnemyIcons();
        this.createActorIcons();
        this.updatePosition();
    };

    Sprite_Timeline.prototype.createTimelineBarSprite = function() {
        this._timelineBarSprite = new Sprite_TimelineBar();
        this.addChild(this._timelineBarSprite);
    };

    Sprite_Timeline.prototype.createEnemyIcons = function() {
        this._enemyIconSprites = $gameTroop.members().map(enemy => {
            // ここでenemyを送ると歩行キャラが表示されないことがある
            // 原因不明
            return new Sprite_EnemyTimelineIcon();
        });
        for (const sprite of this._enemyIconSprites) {
            this.addChild(sprite);
        }
    };

    Sprite_Timeline.prototype.createActorIcons = function() {
        this._actorIconSprites = [];
        for (let i = 0; i < $gameParty.maxBattleMembers(); i++) {
            const sprite = new Sprite_ActorTimelineIcon();
            this._actorIconSprites.push(sprite);
            this.addChild(sprite);
        }
    };

    Sprite_Timeline.prototype.updatePosition = function() {
        this.x = param_Timeline.X || 0;
        this.y = param_Timeline.Y || 0;
    };

    Sprite_Timeline.prototype.update = function() {
        Sprite.prototype.update.call(this);
        this.updateActors();
        this.updateEnemies();
        this.sortChildren();
    };

    Sprite_Timeline.prototype.updateActors = function() {
        const members = $gameParty.battleMembers();
        for (const [i, sprite] of this._actorIconSprites.entries()) {
            sprite.setBattler(members[i]);
        }
    };

    Sprite_Timeline.prototype.updateEnemies = function() {
        const members = $gameTroop.members();
        for (const [i, sprite] of this._enemyIconSprites.entries()) {
            sprite.setBattler(members[i]);
        }
    };

    Sprite_Timeline.prototype.sortChildren = function() {
        const isHorizontal = param_Timeline.Type === 'Horizontal';
        this.children.sort((a, b) => {
            if (a.z !== b.z) {
                return a.z - b.z;
            } else if (isHorizontal && a.x !== b.x) {
                return a.x - b.x;
            } else if (!isHorizontal && a.y !== b.y) {
                return b.y - a.y;
            } else {
                return a.spriteId - b.spriteId;
            }
        });
    };

    //-------------------------------------------------------------------------
    // Window_BattleStatus
    
    // overwrite
    Window_BattleStatus.prototype.placeTimeGauge = function() {};

    //-------------------------------------------------------------------------
    // Scene_Battle

    const _Scene_Battle_update = Scene_Battle.prototype.update;
    Scene_Battle.prototype.update = function() {
        _Scene_Battle_update.apply(this, arguments);
        this.updateTimeline();
    };

    Scene_Battle.prototype.updateTimeline = function() {
        this._timelineSprite.visible = (
            this._messageWindow.isClosed() &&
            !!BattleManager._phase
        );
    };

    const _Scene_Battle_createDisplayObjects = Scene_Battle.prototype.createDisplayObjects;
    Scene_Battle.prototype.createDisplayObjects = function() {
        _Scene_Battle_createDisplayObjects.apply(this, arguments);
        this.createTimelineSprite();
    };

    Scene_Battle.prototype.createTimelineSprite = function() {
        this._timelineSprite = new Sprite_Timeline();
        let index = this.children.indexOf(this._windowLayer);
        if (param_Timeline.Z === 'Before the window') {
            index += 1;
        }
        this.addChildAt(this._timelineSprite, index);
    };

})();
